<?php
use Trustpay\Order;
use Trustpay\Payment;
use Trustpay\PaymentWire;
use Trustpay\PaymentCard;
use Trustpay\PaymentType;
use Trustpay\ResultCodes;

class TrustpayHandler {
    public function __construct() {
        $this->includeLibrary();
    }

    public function getUrl($data) {
        $json['success'] = false;

        $paymentMethod = $this->getPaymentMethod();
        
        if($paymentMethod && $this->validateData($data)) {
            $payment = $this->initPayment($paymentMethod, $data);
            $json['url'] = $payment->getUrl(PaymentType::PURCHASE);
            $json['success'] = true;
        }

        echo json_encode($json);
    }

    public function cancel() {
        $error = false;

        if(isset($_GET['AcquirerResponseId']) && $this->getError($_GET['AcquirerResponseId'])) {
            $error = $this->getError($_GET['AcquirerResponseId']);
        } elseif (isset($_GET['ResultCode']) && $this->getError($_GET['ResultCode'])) {
            $error = $this->getError($_GET['ResultCode']);
        }

        if($error) {
            global $messageStack;
            $messageStack->add_session('checkout_confirmation', $error, 'error');
        }

        zen_redirect(zen_href_link(FILENAME_CHECKOUT_CONFIRMATION, '', 'SSL', true, false));
    }

    public function error() {
        $error = 'Failed';

        if(isset($_GET['AcquirerResponseId']) && $this->getError($_GET['AcquirerResponseId'])) {
            $error = $this->getError($_GET['AcquirerResponseId']);
        } elseif (isset($_GET['ResultCode']) && $this->getError($_GET['ResultCode'])) {
            $error = $this->getError($_GET['ResultCode']);
        }

        global $messageStack;
        $messageStack->add_session('checkout_confirmation', $error, 'error');

        zen_redirect(zen_href_link(FILENAME_CHECKOUT_CONFIRMATION, '', 'SSL', true, false));
    }

    public function notifyWire($data) {
        $this->checkResponse('\Trustpay\ResponseWire', $data, Payment::ENDPOINT_WIRE);
    }

    public function notifyCard($data) {
        $this->checkResponse('\Trustpay\ResponseCard', $data, Payment::ENDPOINT_CARD);
    }

    protected function checkResponse($class, $data, $paymentMethod) {
        $response = new $class($this->getSecretKey());
        $response->initData($data);

        if($response->validateSignature()) {
            $this->changeStatus(
                $response,
                $paymentMethod
            );
        }
    }

    protected function changeStatus(\Trustpay\Response $response, $paymentMethod) {
        if($this->checkComplete($response->getResultCode(), $paymentMethod)) {
            $this->addOrderHistory(
                $response->getReference(),
                MODULE_PAYMENT_TRUSTPAY_COMPLETE_ORDER_STATUS_ID,
                $this->getResultCode($response->getResultCode()),
                $response->getTransactionId()
            );

        } elseif ($this->checkProcessing($response->getResultCode(), $paymentMethod)) {
            $this->addOrderHistory(
                $response->getReference(),
                MODULE_PAYMENT_TRUSTPAY_PROCESING_ORDER_STATUS_ID,
                $this->getResultCode($response->getResultCode()),
                $response->getTransactionId()
            );
        }
    }
    
    protected function getResultCode($code) {
        $resultCode = '';

        if($code == ResultCodes::SUCCESS) {
            $resultCode = 'SUCCESS';
        } elseif ($code == ResultCodes::AUTHORIZED) {
            $resultCode = 'AUTHORIZED';
        } elseif ($code == ResultCodes::PROCESSING) {
            $resultCode = 'PROCESSING';
        }

        return $resultCode;
    }

    protected function checkComplete($resultCode, $paymentMethod) {
        return $resultCode == ResultCodes::SUCCESS || ($resultCode == ResultCodes::AUTHORIZED && $paymentMethod == Payment::ENDPOINT_CARD);
    }

    protected function checkProcessing($resultCode, $paymentMethod) {
        return $resultCode == ResultCodes::PROCESSING || ($resultCode == ResultCodes::AUTHORIZED && $paymentMethod == Payment::ENDPOINT_WIRE);
    }

    protected function addOrderHistory($order_id, $order_status_id, $resultCode, $transaction_id) {
        global $db;

        sleep(5);

        $comment = 'finby response code: ' . $resultCode . ', transaction id: ' . $transaction_id;

        $sql_data_array = array(
            'orders_id' => (int)$order_id,
            'orders_status_id' => (int)$order_status_id,
            'date_added' => 'now()',
            'comments' => zen_db_input($comment),
            'customer_notified' => 0
        );

        zen_db_perform(TABLE_ORDERS_STATUS_HISTORY, $sql_data_array);

        $db->Execute("update " . TABLE_ORDERS  . "
                      set orders_status = '" . (int)$order_status_id . "'
                      where orders_id = '" . (int)$order_id . "'");
    }

    protected function initPayment($paymentMethod, $data) {
        
        $trustPayOrder = new Order();
        $trustPayOrder->setId($data['order_id']);
        //$trustPayOrder->setCurrency($this->getCurrency());
        $trustPayOrder->setCurrency($_SESSION["currency"]);
        $trustPayOrder->setTotal($_SESSION['trustpay_total']);

        if($paymentMethod == 'wire') {
            $payment = new PaymentWire($this->getAccountId(), $this->getSecretKey());
            $notify = 'notifyWire';
        } else {
            $payment = new PaymentCard($this->getAccountId(), $this->getSecretKey());
            $notify = 'notifyCard';

            $trustPayOrder->setBillingCity($_SESSION['trustpay_billing_city']);
            $trustPayOrder->setBillingCountry($_SESSION['trustpay_billing_country']);
            $trustPayOrder->setBillingPostcode($_SESSION['trustpay_billing_postcode']);
            $trustPayOrder->setBillingStreet($_SESSION['trustpay_billing_street']);
            $trustPayOrder->setCardHolder($_SESSION['trustpay_card_holder']);
            $trustPayOrder->setEmail($_SESSION['trustpay_email']);
        }

        $payment->setOrder($trustPayOrder)
            ->setSandbox($this->getSandbox())
            ->setNotificationUrl(HTTPS_SERVER . '/trustpay_handler.php?action=' . $notify)
            ->setReturnUrl(zen_href_link(FILENAME_CHECKOUT_PROCESS, '', 'SSL'))
            ->setCancelUrl(HTTPS_SERVER . '/trustpay_handler.php?action=cancel')
            ->setErrorUrl(HTTPS_SERVER . '/trustpay_handler.php?action=error')
            ->setLocalization($this->getLocale())
            ->setPaymentMethod($paymentMethod)
            ;

        return $payment;
    }

    protected function getPaymentMethod() {
        $method = false;

        if(
            isset($_POST['trustpay_method']) &&
            in_array($_POST['trustpay_method'],
                [
                    Payment::ENDPOINT_WIRE,
                    Payment::ENDPOINT_CARD
                ]
            )
        ) {
            $method = $_POST['trustpay_method'];
        }

        return $method;
    }

    protected function validateData($data) {
        $valid = true;

        $indexes = array(
            'order_id',
        );

        foreach ($indexes as $index) {
            if(empty($data[$index])) {
                $valid = false;
                break;
            }
        }

        if(empty($_SESSION['trustpay_total'])) {
            $valid = false;
        }

        return $valid;
    }

    protected function includeLibrary() {
        $lib_files = array(
            'Order',
            'Payment',
            'PaymentWire',
            'PaymentCard',
            'PaymentType',
            'Response',
            'ResponseCard',
            'ResponseWire',
            'ResultCodes',
            'Signature',
        );

        foreach ($lib_files as $file) {
            require_once 'includes/modules/payment/trustpay/' . $file . '.php';
        }
    }

    protected function getError($code) {
        $translate = '';

        $data = array(
            '100.100.303' => array(
                'en' => 'Rejected - card expired',
                'sk' => 'Odmietnuté – expirovaná karta',
                'cs' => 'Odmítnuto – expirovaná karta'
            ),
            '100.100.600' => array(
                'en' => 'Rejected - invalid CVV',
                'sk' => 'Odmietnuté – neplatné CVV',
                'cs' => 'Odmítnuto – neplatné CVV'
            ),
            '100.100.601' => array(
                'en' => 'Rejected - invalid CVV',
                'sk' => 'Odmietnuté – neplatné CVV',
                'cs' => 'Odmítnuto – neplatné CVV'
            ),
            '100.100.700' => array(
                'en' => 'Rejected - invalid card number',
                'sk' => 'Odmietnuté – neplatné číslo karty',
                'cs' => 'Odmítnuto – neplatné číslo karty'
            ),
            '100.380.401' => array(
                'en' => 'Rejected - failed 3DS authentication',
                'sk' => 'Odmietnuté – neúspešná 3DS autentifikácia',
                'cs' => 'Odmítnuto – neúspěšná 3DS autentifikace'
            ),
            '100.380.501' => array(
                'en' => 'Rejected - failed 3DS authentication',
                'sk' => 'Odmietnuté – neúspešná 3DS autentifikácia',
                'cs' => 'Odmítnuto – neúspěšná 3DS autentifikace'
            ),
            '800.100.151' => array(
                'en' => 'Rejected - invalid card number',
                'sk' => 'Odmietnuté – neplatné číslo karty',
                'cs' => 'Odmítnuto – neplatné číslo karty'
            ),
            '800.100.153' => array(
                'en' => 'Rejected - invalid CVV',
                'sk' => 'Odmietnuté – neplatné CVV',
                'cs' => 'Odmítnuto – neplatné CVV'
            ),
            '800.100.155' => array(
                'en' => 'Rejected - insufficient funds',
                'sk' => 'Odmietnuté – nedostatočné prostriedky',
                'cs' => 'Odmítnuto – nedostatečné prostředky'
            ),
            '800.100.157' => array(
                'en' => 'Rejected - invalid expiry date',
                'sk' => 'Odmietnuté – neplatný dátum expirácie',
                'cs' => 'Odmítnuto – neplatné datum expirace'
            ),
            '800.100.161' => array(
                'en' => 'Rejected - too many invalid tries',
                'sk' => 'Odmietnuté – priveľa neplatných pokusov',
                'cs' => 'Odmítnuto – příliš mnoho neplatných pokusú'
            ),
            '800.100.162' => array(
                'en' => 'Rejected - card limit exceeded',
                'sk' => 'Odmietnuté – prekročený limit karty',
                'cs' => 'Odmítnuto – překročení limitu karty'
            ),
            '800.100.163' => array(
                'en' => 'Rejected - card limit exceeded',
                'sk' => 'Odmietnuté – prekročený limit karty',
                'cs' => 'Odmítnuto – překročení limitu karty'
            ),
            '800.120.101' => array(
                'en' => 'Rejected - card limit exceeded',
                'sk' => 'Odmietnuté – prekročený limit karty',
                'cs' => 'Odmítnuto – překročení limitu karty'
            ),
            '800.120.200' => array(
                'en' => 'Rejected - card limit exceeded',
                'sk' => 'Odmietnuté – prekročený limit karty',
                'cs' => 'Odmítnuto – překročení limitu karty'
            ),
            '800.120.201' => array(
                'en' => 'Rejected - card limit exceeded',
                'sk' => 'Odmietnuté – prekročený limit karty',
                'cs' => 'Odmítnuto – překročení limitu karty'
            ),
            '1001' => array(
                'en' => 'Some of the parameters in request are missing or have invalid format',
                'sk' => 'Some of the parameters in request are missing or have invalid format',
                'cs' => 'Some of the parameters in request are missing or have invalid format'
            ),
            '1002' => array(
                'en' => 'Account with specified ID was not found. Please check that you are using correct AID for playground/live environment.',
                'sk' => 'Account with specified ID was not found. Please check that you are using correct AID for playground/live environment.',
                'cs' => 'Account with specified ID was not found. Please check that you are using correct AID for playground/live environment.'
            ),
            '1003' => array(
                'en' => 'Merchant account has been disabled',
                'sk' => 'Merchant account has been disabled',
                'cs' => 'Merchant account has been disabled'
            ),
            '1004' => array(
                'en' => 'Verification of request signature has failed',
                'sk' => 'Verification of request signature has failed',
                'cs' => 'Verification of request signature has failed'
            ),
            '1005' => array(
                'en' => 'Customer has cancelled the payment',
                'sk' => 'Customer has cancelled the payment',
                'cs' => 'Customer has cancelled the payment'
            ),
            '1006' => array(
                'en' => 'Request was not properly authenticated',
                'sk' => 'Request was not properly authenticated',
                'cs' => 'Request was not properly authenticated'
            ),
            '1007' => array(
                'en' => 'Requested transaction amount is greater than disposable balance',
                'sk' => 'Requested transaction amount is greater than disposable balance',
                'cs' => 'Requested transaction amount is greater than disposable balance'
            ),
            '1008' => array(
                'en' => 'Service cannot be used or permission to use given service has not been granted. If you receive this code, please contact finby for more information.',
                'sk' => 'Service cannot be used or permission to use given service has not been granted. If you receive this code, please contact finby for more information.',
                'cs' => 'Service cannot be used or permission to use given service has not been granted. If you receive this code, please contact finby for more information.'
            ),
            '1009' => array(
                'en' => 'Specified processing ID was already used. Please generate a new PID using Setup payment service call.',
                'sk' => 'Specified processing ID was already used. Please generate a new PID using Setup payment service call.',
                'cs' => 'Specified processing ID was already used. Please generate a new PID using Setup payment service call.'
            ),
            '1010' => array(
                'en' => 'Transaction with specified ID was not found',
                'sk' => 'Transaction with specified ID was not found',
                'cs' => 'Transaction with specified ID was not found'
            ),
            '1011' => array(
                'en' => 'The requested action is not supported for the transaction',
                'sk' => 'The requested action is not supported for the transaction',
                'cs' => 'The requested action is not supported for the transaction'
            ),
            '1014' => array(
                'en' => 'Transaction was rejected by issuer',
                'sk' => 'Transaction was rejected by issuer',
                'cs' => 'Transaction was rejected by issuer'
            ),
            '1100' => array(
                'en' => 'Internal error has occurred. Please contact finby to resolve this issue.',
                'sk' => 'Internal error has occurred. Please contact finby to resolve this issue.',
                'cs' => 'Internal error has occurred. Please contact finby to resolve this issue.'
            ),
            '1101' => array(
                'en' => 'Currency conversion for requested currencies is not supported',
                'sk' => 'Currency conversion for requested currencies is not supported',
                'cs' => 'Currency conversion for requested currencies is not supported'
            )
        );

        if(isset($data[$code]) && isset($data[$code][$this->getLocale()])) {
            $translate = $data[$code][$this->getLocale()];
        }

        return $translate;
    }

    private function getCurrency() {
        return MODULE_PAYMENT_TRUSTPAY_CURRENCY;
    }

    private function getAccountId() {
        return $this->getSandbox() ? MODULE_PAYMENT_TRUSTPAY_TEST_ACCOUNT_ID : MODULE_PAYMENT_TRUSTPAY_LIVE_ACCOUNT_ID;
    }

    private function getSecretKey() {
        return $this->getSandbox() ? MODULE_PAYMENT_TRUSTPAY_TEST_SECRET_KEY : MODULE_PAYMENT_TRUSTPAY_LIVE_SECRET_KEY;
    }

    private function getSandbox() {
        return MODULE_PAYMENT_TRUSTPAY_SANDBOX == 'Test' ? true : false;
    }

    private function getLocale() {
        return MODULE_PAYMENT_TRUSTPAY_LOCALISATION;
    }
    
}

ini_set('display_errors', 0);
header('Cache-Control: no-cache, no-store, must-revalidate');

require 'includes/application_top.php';

if(isset($_GET['action'])) {
    $handler = new TrustpayHandler();

    switch ($_GET['action']) {
        case 'get_url':
            $handler->getUrl($_POST);
            break;
        case  'notifyWire':
            $handler->notifyWire($_GET);
            break;
        case  'notifyCard':
            $handler->notifyCard($_GET);
            break;
        case 'cancel':
            $handler->cancel();
            break;
        case 'error':
            $handler->error();
            break;
    }
}