<?php


namespace TrustPay\PaymentGateway\Model;

use Magento\Checkout\Model\ConfigProviderInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Locale\Resolver;
use TrustPay\PaymentGateway\Helper\Data as Helper;

/**
 * Class TrustPayConfigProvider
 * @package TrustPay\PaymentGateway\Model
 */
class TrustPayConfigProvider implements ConfigProviderInterface
{
    /**
     * @var Helper
     */
    protected $helper;

    /**
     * @var CheckoutSession
     */
    protected $checkoutSession;

    /**
     * @var Resolver
     */
    protected $store;

    /**
     * @var EventManager
     */
    protected $eventManger;

    /**
     * TrustPayConfigProvider constructor.
     * @param Helper $helper
     * @param CheckoutSession $checkoutSession
     * @param Resolver $store
     * @param EventManager $eventManger
     */
    public function __construct(
        Helper $helper,
        CheckoutSession $checkoutSession,
        Resolver $store,
        EventManager $eventManger
    ) {
        $this->helper = $helper;
        $this->checkoutSession = $checkoutSession;
        $this->store = $store;
        $this->eventManger = $eventManger;
    }

    /**
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getConfig()
    {
        $configData = [
            'trust_payment_config' => [
                'checkout_disclaimer' => $this->helper->getConfigHelper()->getCheckoutDisclaimer(),
            ]
        ];
        
        $configData['trust_payment_config']['gateways'] = [];
        
        if(in_array($this->getQuote()->getQuoteCurrencyCode(), $this->helper->getCardAllowedCurrencies())) {
            $configData['trust_payment_config']['gateways'][] = [
                        'code' => Helper::CARD_GATEWAY_CODE,
                        'label' => $this->helper->getConfigHelper()->getCardsPaymentsTitle(),
                        'image' => Helper::CARD_GATEWAY_CODE . '.png',
                        'iframeUrl' => $this->helper->getCallbackApiUrl(Helper::CARD_GATEWAY_CODE), //$this->getIFrameUrl(Helper::CARD_GATEWAY_CODE),
                        'isDefaultChecked' => false,
                        'isRedirect' => true
                    ];
        }
        
        if(in_array($this->getQuote()->getQuoteCurrencyCode(), $this->helper->getWireAllowedCurrencies())) {
            $configData['trust_payment_config']['gateways'][] = [
                        'code' => Helper::WIRE_GATEWAY_CODE,
                        'label' => $this->helper->getConfigHelper()->getBankTransfersTitle(),
                        'image' => Helper::WIRE_GATEWAY_CODE . '.png',
                        'iframeUrl' => $this->helper->getCallbackApiUrl(Helper::WIRE_GATEWAY_CODE),//$this->getIFrameUrl(Helper::WIRE_GATEWAY_CODE),
                        'isDefaultChecked' => true,
                        'isRedirect' => true
                    ];
        }
        
        $this->eventManger->dispatch(
            'trustpay_payment_gateway_get_checkout_config_after',
            ['checkoutConfigData' => $configData]
        );
        return $configData;
    }

    /**
     * @param string $gatewayCode
     * @return string
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getIFrameUrl(string $gatewayCode): string
    {
        return $this->helper->getEndpointUrl($gatewayCode) . '/?' . http_build_query($this->getQueryParams($gatewayCode));
    }

    /**
     * @param $gatewayCode
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getQueryParams(string $gatewayCode): array
    {
        $requiredParams = $this->getRequiredApiParams();
        $additionalParams = $this->getAdditionalApiParams($gatewayCode);

        if ($gatewayCode === Helper::CARD_GATEWAY_CODE) {
            $requiredParams = array_merge($requiredParams, $this->getRequiredCardApiParams());
        }

        $this->eventManger->dispatch(
            'trustpay_payment_gateway_checkout_config_get_api_query_params_after',
            [
                'api_required_params' => $requiredParams,
                'api_additional_params' => $additionalParams,
                'gateway_code' => $gatewayCode
            ]
        );
        $requiredParams['Signature'] = (string)$this->helper->getSignature($requiredParams);
        return array_merge($requiredParams, $additionalParams);
    }

    /**
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getRequiredApiParams(): array
    {
        return [
            'AccountId' => (string)$this->helper->getAccountId(),
            'Amount' => number_format($this->getQuote()->getGrandTotal(), 2, '.', ''),
            'Currency' => (string)$this->getQuote()->getQuoteCurrencyCode(),
            'Reference' => (string)$this->getQuote()->getId(),
            'PaymentType' => Helper::PAYMENT_TYPE_PURCHASE
        ];
    }

    /**
     * @return array
     */
    protected function getAdditionalApiParams($gateway): array
    {
        return [
            'ReturnUrl' => $this->helper->getCallbackReturnUrl(),
            'CancelUrl' => $this->helper->getCallbackCancelUrl(),
            'ErrorUrl' => $this->helper->getCallbackErrorUrl(),
            'Url' => $this->helper->getCallbackErrorUrl(),
            'Localization' => strstr($this->store->getLocale(), '_', true),
            'PluginVersion' => TRUSTPAY_VERSION,
            'NotificationUrl' => $this->helper->getCallbackNotificationUrl($gateway),
            'IsRedirect' => 'true'
        ];
    }

    /**
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getRequiredCardApiParams(): array
    {
        return [
            'BillingCity' => $this->getQuote()->getBillingAddress()->getCity(),
            'BillingCountry' => $this->getQuote()->getBillingAddress()->getCountry(),
            'BillingPostcode' => $this->getQuote()->getBillingAddress()->getPostcode(),
            'BillingStreet' => $this->getQuote()->getBillingAddress()->getStreetFull(),
            'CardHolder' => $this->getQuote()->getBillingAddress()->getName(),
            'Email' => $this->getQuote()->getBillingAddress()->getEmail()
        ];
    }

    /**
     * @return \Magento\Quote\Model\Quote
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getQuote()
    {
        return $this->checkoutSession->getQuote();
    }
}
