<?php


namespace TrustPay\PaymentGateway\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;

/**
 * Class Data
 * @package TrustPay\PaymentGateway\Helper
 */
class Data extends AbstractHelper
{
    const PAYMENT_CARD_LIVE_API_ENDPOINT_MERCHANT_PORTAL = "https://amapi.finby.eu/mapi5/Card/PayPopup";

    const PAYMENT_WIRE_LIVE_API_ENDPOINT_MERCHANT_PORTAL = 'https://amapi.finby.eu/mapi5/wire/paypopup';

    const WIRE_GATEWAY_CODE = 'wire';
    const CARD_GATEWAY_CODE = 'card';

    const PAYMENT_TYPE_PURCHASE = 0;

    /**
     * @var Config
     */
    private $configHelper;

    /**
     * @var array
     */
    protected $allowedCurrencies = ['EUR', 'CZK'];
    
    /** NEdokoncene - tu sa to neda spravit, to by bolo treba nejako cely plugin prerobit */
    private $cardAllowedCurrencies = [
        'AUD',
        'CAD',
        'CZK',
        'DKK',
        'EUR',
        'GBP',
        'HRK',
        'HUF',
        'JPY',
        'NOK',
        'PLN',
        'RON',
        'SEK',
        'USD',
    ];

    /**
     * Data constructor.
     * @param Context $context
     * @param Config $configHelper
     */
    public function __construct(Context $context, Config $configHelper)
    {
        $this->configHelper = $configHelper;
        parent::__construct($context);
    }

    /**
     * @return string
     */
    public function getAccountId()
    {
        return $this->getConfigHelper()->getIsSandBoxModeEnabled()
            ? $this->getConfigHelper()->getSandBoxAccountId()
            : $this->getConfigHelper()->getLiveAccountId();
    }

    /**
     * @return string
     */
    public function getSecretKey()
    {
        return $this->getConfigHelper()->getIsSandBoxModeEnabled()
            ? $this->getConfigHelper()->getSandBoxSecretKey()
            : $this->getConfigHelper()->getLiveSecretKey();
    }

    /**
     * @param string $gateway
     * @return bool|string
     */
    public function getEndpointUrl(string $gateway)
    {
        switch ($gateway) {
            case self::CARD_GATEWAY_CODE:
                return self::PAYMENT_CARD_LIVE_API_ENDPOINT_MERCHANT_PORTAL;
                break;
            case self::WIRE_GATEWAY_CODE:
                return self::PAYMENT_WIRE_LIVE_API_ENDPOINT_MERCHANT_PORTAL;
                break;
            default:
                return false;
        }
    }

    /**
     * @return string
     */
    public function getCallbackErrorUrl()
    {
        return $this->_getUrl('trustpay/callback/index');
    }

    /**
     * @return string
     */
    public function getCallbackCancelUrl()
    {
        return $this->_getUrl('trustpay/callback/index');
    }

    /**
     * @return string
     */
    public function getCallbackReturnUrl()
    {
        return $this->_getUrl('trustpay/callback/success');
    }
    
    /**
     * @return string
     */
    public function getCallbackApiUrl($gateway)
    {
        return $this->_getUrl('trustpay/callback/apiurl', ['gateway' => $gateway]);  
    }
    
    /**
     * @return string
     */
    public function getCallbackNotificationUrl($gateway)
    {
        return $this->_getUrl('trustpay/callback/notification', ['gateway' => $gateway]);  
    }

    /**
     * @param array $data
     * @return string
     */
    public function getSignature(array $data)
    {
        return strtoupper(
            hash_hmac(
                'sha256',
                pack('A*', $this->prepareMessageForSignature($data)),
                pack('A*', $this->getSecretKey())
            )
        );
    }

    /**
     * @return Config
     */
    public function getConfigHelper()
    {
        return $this->configHelper;
    }

    /**
     * @param array $data
     * @return string
     */
    protected function prepareMessageForSignature(array $data)
    {
        return implode('/', $data);
    }

    /**
     * @return array
     */
    public function getAllowedCurrencies()
    {
        //return $this->allowedCurrencies;
         return array_unique(array_merge($this->allowedCurrencies, $this->cardAllowedCurrencies));
    }
    
    /**
     * @return array
     */
    public function getCardAllowedCurrencies()
    {
        return $this->cardAllowedCurrencies;
    }
    
    public function getWireAllowedCurrencies()
    {
        return $this->allowedCurrencies;
    }
}
